module laplacian_mod
  use CSR_mod
  implicit none
  contains
    !> contruction of the Laplacian matrix
    !! using a 5-points finite difference scheme
    !! @param A CSR matrix
    !! @param nx number of points in x direction
    !! @param ny number of points in y direction
    !! @param dx space step in x direction
    !! @param dy space step in y direction
    subroutine LaplacianMatrix(A, nx, ny, dx, dy)  
      TYPE(CSR) :: A
      INTEGER :: nx, ny, nnz
      INTEGER :: ind, i, j, ii
      REAL(kind=8) :: dx, dy, dx2, dy2

      dx2 = 1.d0/dx/dx
      dy2 = 1.d0/dy/dy

      nnz = 5*(nx-2)*(ny-2)+8*(nx-2)+8*(ny-2)+12

      call createCSR(A, nx*ny, nnz)

      A%row_ptr(1) = 1
      do j = 1, ny
         do i = 1, nx
            ii = i+(j-1)*nx
            ind = A%row_ptr(ii)
            
            if (j-1>=1) then
               A%col_ind(ind) = ii-nx
               A%val(ind) = -dy2
               ind = ind + 1
            end if

            if (i-1>=1) then
               A%col_ind(ind) = ii-1
               A%val(ind) = -dx2
               ind = ind + 1               
            end if

            A%col_ind(ind) = ii
            A%val(ind) = 2.d0*dx2 + 2.d0*dy2
            ind = ind + 1               

            if (i+1<=nx) then
               A%col_ind(ind) = ii+1
               A%val(ind) = -dx2
               ind = ind + 1               
            end if

            if (j+1<=ny) then
               A%col_ind(ind) = ii+nx
               A%val(ind) = -dy2
               ind = ind + 1
            end if

            A%row_ptr(ii+1) = ind 
         end do
      end do
    end subroutine LaplacianMatrix
    
    !> set Dirichlet condition in the second member vector
    !!
    !! u = 1. on the boundary
    !! @param x vector
    !! @param nx number of points in x direction
    !! @param ny number of points in y direction
    !! @param dx space step in x direction
    !! @param dy space step in y direction
    subroutine setDirichlet(x, nx, ny, dx, dy)
      real(kind=8), pointer, dimension(:) :: x
      INTEGER :: nx, ny
      INTEGER :: i, j
      REAL(kind=8) :: dx, dy, dx2, dy2

      dx2 = 1.d0/dx/dx
      dy2 = 1.d0/dy/dy
      
      x(1:nx) = x(1:nx) + dx2
      x(nx*(ny-1)+1:) = x(nx*(ny-1)+1:) + dx2
      x(1::nx) = x(1::nx) + dy2
      x(nx::nx) = x(nx::nx) + dy2
    end subroutine setDirichlet
end module laplacian_mod
